$Title Cascade Reservoir Optimal Operation Model
$eolcom //
$onempty
$onundf
$offupper

* ====================== SET DEFINITIONS AND DATA LOADING ======================

* Define basic sets for data structure
Set r(*),r2(*),r3(*),r4(*),r5(*),time_long1(*),time_long3(*),time_long4(*),time_long5(*),
    time_long "Time series from t1 to t731" /t1*t731/
    Index(*);

* Load inflow data from GDX file
$gdxin all_inflow_qujian.gdx
$load time_long1 r
Parameter inflow_total(time_long, r);
$load inflow_total
$gdxin

* Load reservoir basic parameters from GDX file
$gdxin all_basic_Data.gdx
$load r2 Index
Parameter X(r, Index);
$Load X
$gdxin

* Load water diversion project data from GDX file
$gdxin water_diversion_of_A_and_B.gdx
$load r3 time_long3
Parameter water_division_AND_real_h(time_long,r3);
$Load water_division_AND_real_h
$gdxin

* Load reservoir water loss data from GDX file
$gdxin water_loss_of_reservoir.gdx
$load r4 time_long4
Parameter water_loss(time_long,r);
$Load water_loss
$gdxin

* ====================== RESERVOIR CLASSIFICATION ======================

* Classify reservoirs by type for different operational constraints
set 
   r_heyu(r)                   "Cascade power stations in the system"      /A,B,C,D/
   r_Runoff(r)                 "Run-of-river power stations (no storage)"        /B/               
   r_daily(r)                  "Daily regulation power stations"        /C,D/
;

* ====================== RESERVOIR PARAMETERS ======================

* Define all reservoir operational parameters
parameters
   Nmax(r)                  "Installed capacity (10^4 kW)"      //
   Nfirm(r)                 "Firm output (10^4 kW)"     //
   Qmin(r)                  "Minimum flow requirement (m3/s)"     //
   Qmax(r)                  "Maximum flow capacity (m3/s)"     //
   Guarantee_rate(r)        "Reliability rate (decimal)"       //
   V0(r)                    "Initial storage volume (10^4 m3)"     //
   V_end(r)                 "Final storage volume (10^4 m3)"
   Vmax(r)                  "Normal storage level (10^4 m3)"     //
   Vmin(r)                  "Dead storage level (10^4 m3)"       //
   Eta(r)                   "Efficiency coefficient"           //
   H0(r)                    "Initial water level (m)"      //
   H_xia(r)                 "Downstream water level (m)"      //
   add_tax(r)               "Value-added tax rate (decimal)"  //
   payed_tax(r)             "Electricity generation fees including funds and water resource tax (yuan/kWh)" //
   electric_price(r)        "Electricity price (yuan/kWh)"      //
   divison_price(r)         "Water diversion compensation rate" //
   judge(time_long)         "Time period judgment: 1 for decade, 0 for day" //
   A_division(time_long)  "Water diversion project A"        //
   B_division(time_long)   "Water diversion project B"          //
   real_h(time_long)        "Actual hours per calculation period"
;

* Extract parameter values from loaded data matrix X
Nmax(r) = X(r,'Nmax');      // Extract installed capacity
Nfirm(r) = X(r,'Nfirm');    // Extract firm output
Qmin(r) = X(r,'Qmin');      // Extract minimum flow
Qmax(r) = X(r,'Qmax');      // Extract maximum flow
Guarantee_rate(r) = X(r,'Guarantee_rate'); // Extract guarantee rate
V0(r) = X(r,'V0');          // Extract initial storage
V_end(r)=X(r,'V_end');
Vmax(r) = X(r,'Vmax');      // Extract normal storage
Vmin(r) = X(r,'Vmin');      // Extract dead storage
Eta(r) = X(r,'Eta');        // Extract efficiency coefficient
H0(r) = X(r,'H0');          // Extract initial water level
H_xia(r) = X(r,'H_xia');    // Extract downstream water level
add_tax(r) = X(r,'add_tax'); // Extract value-added tax
payed_tax(r) = X(r,'payed_tax'); // Extract electricity generation fees
divison_price(r) = X(r,'divison_price'); // Extract water diversion compensation rate
electric_price(r) = X(r,'electric_price'); // Extract electricity price

* Extract water diversion and time parameters
A_division(time_long) = water_division_AND_real_h(time_long,'Division_A'); // Extract water diversion A
B_division(time_long) = water_division_AND_real_h(time_long,'Division_B'); // Extract water diversion B
real_h(time_long)=water_division_AND_real_h(time_long,'real_time');// Extract actual hours per period
judge(time_long) = water_division_AND_real_h(time_long,'judge'); // Extract time period judgment

* ====================== SCALAR PARAMETERS ======================

* Define physical constants and operational coefficients
scalar          
   Secs_hour                            "Seconds per hour" / 3600 /        
   Hour_day                             "Hours per day" /24/   
   M                                    "Big M coefficient for binary logic" / 4000 /          
   epsilon                              "Small positive number for numerical stability" / 1e-5 /          
   price_diversion3                     "Diversion project 3 price" /0.0915/
   Pipeline_flow_capacity               "Pipeline flow capacity"    /77/
   A_small_capacity                    "Small unit generation capacity of Reservoir A"  /2.4/
   closed_zone_up                       "Upper limit of non-generation zone" /80/
   closed_zone_down                     "Lower limit of non-generation zone" /52/
   M_head                            "Big-M value for head interval constraints (m)" / 100 /
   ;

* ====================== VARIABLE DECLARATIONS ======================

* Define positive variables (non-negative)
positive variables
   Q(time_long,r_heyu)                 "Generation flow (m3/s)"
   Q_dy(time_long)                "Water diversion to E when A has spillage (m3/s)"
   Spill(time_long,r_heyu)             "Spillage flow (m3/s)"
   Spill_W(time_long,r_heyu)           "Spillage volume (10^6 m3)"
   V(time_long,r_heyu)                 "Initial storage of period (10^4 m3)"
   V_end_t(time_long,r_heyu)           "Final storage of period (10^4 m3)"
   H_up(time_long,r_heyu)              "Upstream water level (m)"
   H0_up(time_long,r_heyu)             "Initial upstream water level (m)"
   H1_UP(time_long,r_heyu)             "Final upstream water level (m)"
   H_down(time_long,r_heyu)            "Downstream water level (m)"
   Water_H_head(time_long,r_heyu)      "Head difference between upstream and downstream (m)"
   Inflow(time_long,r_heyu)            "Reservoir inflow"
   E(time_long,r_heyu)                 "Energy generation (10^4 kWh)"
   Compensatory_division(time_long,r_heyu)   "Uncompensated flow during spillage (m3/s)"
   Compensatory_division_fee(time_long,r_heyu) "Uncompensated water volume cost (10^4 yuan)"
   E_profit(time_long,r_heyu)          "Generation benefit"
   N(time_long,r_heyu)                 "Average power output (10^4 kW)"
   time_xun_or_day(time_long)     "Specific days in the period"
   Q_DIVERSION3_A(time_long)            "Water diversion from F to A (m3/s)"
   Q_DIVERSION3_E(time_long)            "Water diversion from F to E (m3/s)"
   Q_small_A(time_long)          "Small unit generation flow of A (m3/s)"
   Q_large_A(time_long)          "Large unit generation flow of A (m3/s)"
   division_and_loss(time_long,r_heyu)"Water loss due to diversion and evaporation/seepage (m3/s)"
   ;
   
* Define binary variables for logical constraints
binary variable
   abandon_or_not(time_long,r_heyu)      "Indicator for spillage at A (1=spillage, 0=no spillage)"
   A_small_stage(time_long)           "Indicator for small generation stage [0,52] at A"
   A_large_stage(time_long)          "Indicator for large generation stage [80,516] at A"
   E_zone1(time_long)            "Indicator for head interval 1 at E: 12.5~40m"
   E_zone2(time_long)            "Indicator for head interval 2 at E: 42~70m"
   judge_generate(time_long)      "Indicator for generation capability below dead level at A"

;

* Define free variable for objective function
free variable 
   Z "Total benefit (10^4 kWh)";

* ====================== EQUATION DECLARATIONS ======================

* Reservoir operational equations
Equations
   WaterBalance(r_heyu,time_long)             "Water balance equation constraint" 
   DefV_end_t(r_heyu,time_long)               "Final storage definition"
   DailyDischarge_C(time_long)             "Daily regulation discharge rule for Reservoir C"
   DailyDischarge_D(time_long)              "Daily regulation discharge rule for Reservoir D"
   RunoffDischarge(time_long)                 "Run-of-river power station discharge rule"
   StorageLowerBound(r_heyu,time_long)        "Dead storage constraint"
   StorageUpperBound(r_heyu,time_long)        "Normal storage constraint"
   FlowLowerBound(r_heyu,time_long)           "Minimum generation flow constraint"
   FlowUpperBound(r_heyu,time_long)           "Maximum generation flow constraint"
   HeadCalc_up(time_long,r_heyu)              "Upstream water level calculation"
   H0_Calc_up(time_long,r_heyu)               "Initial upstream water level calculation"
   H1_Calc_up(time_long,r_heyu)               "Final upstream water level calculation"
   H_head_Calc(time_long,r_heyu)             "Head difference calculation"
   HeadCalc_down(r_heyu,time_long)            "Downstream water level calculation"
   Energy_each_one(r_heyu,time_long)          "Energy generation calculation"
   PowerCalc(r_heyu,time_long)                "Power output calculation"
   PowerLimit(r_heyu,time_long)               "Power output upper limit constraint" 
   InitialVol(r_heyu)                         "Initial storage constraint"
   Inflow_define(time_long,r_heyu)            "Reservoir inflow definition"
   Time_transformation(time_long)             "Time transformation between decade and day periods"
   // Water diversion related equations    
   DIVERSION1_and_C_keep(time_long)           "Ensure Reservoir C has sufficient inflow"
   DIVERSION1_and_D_keep(time_long)           "Ensure Reservoir D has sufficient inflow"
   total_dibate(time_long,r_heyu)        "Total water loss due to diversion and evaporation/seepage (m3/s)" 
   // Unit splitting equations for Reservoir A
   Split_flow_A(time_long)              "Total generation flow splitting for Reservoir A"
   Small_flow_limit(time_long)           "Small unit flow limit"
   Large_flow_limit(time_long)           "Large unit flow limit"
   Large_flow_min(time_long)             "Large unit minimum flow constraint"
   judge_r_spill_1(time_long)            "Spillage detection constraint 1"
   judge_r_spill_2(time_long)            "Spillage detection constraint 2"
;

* Unit operation mode selection equations for Reservoir A
Equations
   Differentiate_intervals(time_long)    "Generation interval differentiation constraint"
   p_inst_cap(time_long)                 "Unit capacity constraints for different operation modes"
;

* Economic calculation equations
Equations
   E_Profit_cap(time_long,r_heyu)        "Generation benefit calculation per period"
   TotalEnergy                           "Total profit objective function"
   Water_Compensatory_fee(time_long,r_heyu)"Water compensation cost function - 10^4 m3"
   Water_Compensatory(time_long,r_heyu)  "Water compensation volume function"
;

* ====================== EQUATION DEFINITIONS ======================

* Time period length calculation
Time_transformation(time_long)..
    time_xun_or_day(time_long) =e= real_h(time_long);

* Final storage volume definition
* This equation links the storage at the end of current period to the beginning of next period
DefV_end_t(r_heyu,time_long)..
    V_end_t(time_long,r_heyu) =e= 
        V(time_long+1,r_heyu) $ (ord(time_long) < card(time_long))  // For non-final periods
        + (V0(r_heyu)      $ (ord(time_long) = card(time_long)))$(not sameas(r_heyu,'A')) // For final period of other reservoirs
        + (V_end(r_heyu)   $ (ord(time_long) = card(time_long)))$(sameas(r_heyu,'A')); // For final period of Reservoir A

* Reservoir inflow definition
* Calculates inflow to each reservoir considering upstream releases and diversions
Inflow_define(time_long,r_heyu)..
    Inflow(time_long,r_heyu) =e= 
        (Inflow_total(time_long,'A'))$(sameas(r_heyu,'A'))  // Reservoir A: natural inflow only
        +(Inflow_total(time_long,'B')+Q(time_long,'A') + Spill(time_long, 'A')-B_division(time_long))$(sameas(r_heyu,'B'))  // Reservoir B: natural inflow + A's releases - diversion B
        +(Inflow_total(time_long,'C')+Q(time_long,'B') + Spill(time_long, 'B'))$(sameas(r_heyu,'C'))  // Reservoir C: natural inflow + B's releases
        +(Inflow_total(time_long,'D')+Q(time_long,'C') + Spill(time_long, 'C'))$(sameas(r_heyu,'D'));  // Reservoir D: natural inflow + C's releases

* Water balance and storage continuity equation
* Fundamental equation: Storage change = Inflow - Outflow - Losses
WaterBalance(r_heyu,time_long)..
      V(time_long,r_heyu) + (Inflow(time_long,r_heyu) - Q(time_long,r_heyu) - Spill(time_long,r_heyu)-A_division(time_long)$(sameas(r_heyu,'A'))-water_loss(time_long,r_heyu)) * time_xun_or_day(time_long)*Secs_hour/1e4 =e= V_end_t(time_long,r_heyu);

* Special constraints for daily regulation reservoirs
* These reservoirs must release all inflow in the same period
DailyDischarge_C(time_long).. 
    Q(time_long,'C') + Spill(time_long,'C') =e= Inflow(time_long,'C')-water_loss(time_long,'C');
DailyDischarge_D(time_long).. 
    Q(time_long,'D') + Spill(time_long,'D') =e= Inflow(time_long,'D')-water_loss(time_long,'D');

* Special constraint for run-of-river reservoir
* No storage capacity, all inflow must be released
RunoffDischarge(time_long)..Q(time_long,'B') + Spill(time_long,'B')+water_loss(time_long,'B') =e= Inflow(time_long,'B');

* Physical constraints on storage and flow
StorageLowerBound(r_heyu,time_long).. V(time_long,r_heyu) =g= Vmin(r_heyu);  // Minimum storage level
StorageUpperBound(r_heyu,time_long).. V(time_long,r_heyu) =l= Vmax(r_heyu);  // Maximum storage level
FlowLowerBound(r_heyu,time_long).. Q(time_long,r_heyu) =g= Qmin(r_heyu);     // Minimum generation flow
FlowUpperBound(r_heyu,time_long).. Q(time_long,r_heyu) =l= Qmax(r_heyu);     // Maximum generation flow

* Storage-water level relationship calculations
* Convert storage volume to water level using empirical formulas
H0_Calc_up(time_long,r_heyu)..
H0_up(time_long,r_heyu) =e=        
      (21.531*log(V(time_long,r_heyu))+35.146)$(sameas(r_heyu,'A'))  // Reservoir A: logarithmic relationship
   + (0.009*(V(time_long,r_heyu)) + 217.97)$(sameas(r_heyu,'B'))  // Reservoir B: linear relationship
   + (0.0011*(V(time_long,r_heyu)) + 211.1)$(sameas(r_heyu,'C'))  // Reservoir C: linear relationship
   + (0.0008*(V(time_long,r_heyu)) + 178.374437499997)$(sameas(r_heyu,'D'));  // Reservoir D: linear relationship

H1_Calc_up(time_long,r_heyu)..
H1_UP(time_long,r_heyu) =e=
     (21.531*log(V_end_t(time_long,'A'))+35.146)$(sameas(r_heyu,'A')) // Reservoir A
   + (0.009*(V_end_t(time_long,r_heyu)) + 217.97)$(sameas(r_heyu,'B'))  // Reservoir B
   + (0.0011*(V_end_t(time_long,r_heyu)) + 211.1)$(sameas(r_heyu,'C'))  // Reservoir C
   + (0.0008*(V_end_t(time_long,r_heyu)) + 178.374437499997)$(sameas(r_heyu,'D'));  // Reservoir D

* Average upstream water level during the period
HeadCalc_up(time_long,r_heyu)..
   H_up(time_long,r_heyu) =e= 0.5*(H0_up(time_long,r_heyu)+H1_UP(time_long,r_heyu));

* Downstream water level calculation (assumed constant for simplicity)
HeadCalc_down(r_heyu,time_long)..
   H_down(time_long,r_heyu) =e=
     242.9$(sameas(r_heyu,'A'))   // Reservoir A
   + 215.5$(sameas(r_heyu,'B'))  // Reservoir B
   + 191.5$(sameas(r_heyu,'C'))  // Reservoir C
   + 153.4$(sameas(r_heyu,'D'));  // Reservoir D

* Head difference calculation (driving force for power generation)
H_head_Calc(time_long,r_heyu)..Water_H_head(time_long,r_heyu) =e=H_up(time_long,r_heyu)-H_down(time_long,r_heyu);

* Energy generation calculation
* Different formulas for different reservoir types based on their characteristics
Energy_each_one(r_heyu,time_long)..
    E(time_long,r_heyu) =e=
     (Q(time_long,r_heyu)*time_xun_or_day(time_long)*Secs_hour/(0.0117*sqr(H_up(time_long,r_heyu))-7.1449*H_up(time_long,r_heyu)+1098)/1e4)$(sameas(r_heyu,'A'))            // Reservoir A: empirical formula
    +(Q(time_long,r_heyu)*time_xun_or_day(time_long)*(Eta(r_heyu)*9.81*(H_up(time_long,r_heyu)-H_down(time_long,r_heyu)))/1e4)$((not sameas(r_heyu,'A')) and (not sameas(r_heyu,'C')) and (not sameas(r_heyu,'D')))  // Other reservoirs: theoretical formula
    +(Q(time_long,r_heyu)*time_xun_or_day(time_long)*Secs_hour/16.21/1e4)$(sameas(r_heyu,'C')) // Reservoir C: fixed water consumption rate
    +(Q(time_long,r_heyu)*time_xun_or_day(time_long)*Secs_hour/11.82/1e4)$(sameas(r_heyu,'D'));     // Reservoir D: fixed water consumption rate

* Power output calculation
PowerCalc(r_heyu,time_long).. 
    N(time_long,r_heyu) =e= E(time_long,r_heyu)/(time_xun_or_day(time_long)+1e-10);

* Power output upper limit constraint
PowerLimit(r_heyu,time_long).. 
    N(time_long,r_heyu) =l= Nmax(r_heyu);

* Initial storage constraint
InitialVol(r_heyu).. 
    V('t1',r_heyu) =e= V0(r_heyu);

* Spillage detection constraints using binary variables
judge_r_spill_1(time_long)..Spill(time_long,'A') =l= 3600*abandon_or_not(time_long,'A');
judge_r_spill_2(time_long)..Spill(time_long,'A') =g= 1e-4*abandon_or_not(time_long,'A');

* ====================== RESERVOIR A UNIT SPLITTING ======================

* Reservoir A has two types of units: small and large with different characteristics
Split_flow_A(time_long)..
    Q(time_long,'A') =e= Q_small_A(time_long) + Q_large_A(time_long);

* Small unit flow limit: 0-52 m3/s
Small_flow_limit(time_long)..
    Q_small_A(time_long) =l= closed_zone_down;

* Large unit flow limit (only active when in large stage)
Large_flow_limit(time_long)..
    Q_large_A(time_long) =l= (Qmax('A') - closed_zone_down) * A_large_stage(time_long);

* Large unit minimum flow constraint
Large_flow_min(time_long)..
    Q_large_A(time_long) =g= (closed_zone_up-closed_zone_down) * A_large_stage(time_long);
    
* Generation stage selection for Reservoir A
* Only one stage can be active at a time
Differentiate_intervals(time_long).. 
    A_small_stage(time_long) + A_large_stage(time_long) =l= 2;

* Installed capacity constraint based on operation stage
p_inst_cap(time_long).. 
    N(time_long,'A') =l= A_small_capacity + (Nmax('A')-2.4) * A_large_stage(time_long);

* ====================== WATER DIVERSION CONSTRAINTS ======================

* Ensure sufficient inflow for downstream reservoirs considering water diversion
DIVERSION1_and_C_keep(time_long)..Inflow(time_long,'C') =g= water_loss(time_long,'C');
DIVERSION1_and_D_keep(time_long)..Inflow(time_long,'D') =g= water_loss(time_long,'D');
total_dibate(time_long,r_heyu)..division_and_loss(time_long,r_heyu) =e= water_loss(time_long,r_heyu)+A_division(time_long)$(sameas(r_heyu,'A'));

* ====================== ECONOMIC CALCULATIONS ======================

* Water compensation calculation for different reservoirs
Water_Compensatory(time_long,r_heyu)..
    Compensatory_division(time_long,r_heyu) =e=      
       ((A_division(time_long))$(sameas(r_heyu,'A')) // Reservoir A compensation
      + ((A_division(time_long)+B_division(time_long)))$(sameas(r_heyu,'B'))   // Reservoir B compensation
      + ((A_division(time_long)+B_division(time_long)))$(sameas(r_heyu,'C')) // Reservoir C compensation
      + ((A_division(time_long)+B_division(time_long)))$(sameas(r_heyu,'D'))); // Reservoir D compensation

* Water compensation cost calculation
Water_Compensatory_fee(time_long,r_heyu)..
   Compensatory_division_fee(time_long,r_heyu) =e= Compensatory_division(time_long,r_heyu)*time_xun_or_day(time_long)*Secs_hour/1e4*divison_price(r_heyu);

* Generation benefit calculation with different tax rates for different units
E_Profit_cap(time_long,r_heyu)..
    E_profit(time_long,r_heyu) =e= 
        // Reservoir A: Different tax rates for small and large units
        E(time_long,r_heyu)*(Q_small_A(time_long)/(Q(time_long,r_heyu)+5e-5) * (electric_price(r_heyu)/(1+add_tax(r_heyu)-0.1)-payed_tax(r_heyu)) 
        + Q_large_A(time_long)/(Q(time_long,r_heyu)+5e-5) * (electric_price(r_heyu)/(1+add_tax(r_heyu))-payed_tax(r_heyu))
        )$(sameas(r_heyu,'A'))
        // Other reservoirs: Standard tax rate
        + (E(time_long,r_heyu) * (electric_price(r_heyu)/(1+add_tax(r_heyu))-payed_tax(r_heyu)))$(not sameas(r_heyu,'A'));

* Objective function: Maximize total benefit (generation profit + water diversion compensation)
TotalEnergy..
    Z =e= sum((time_long,r_heyu), E_profit(time_long,r_heyu))
       + sum((time_long,r_heyu), Compensatory_division_fee(time_long,r_heyu));

* ====================== MODEL SOLUTION CONFIGURATION ======================

* Define the complete model
model Reservoir /all/;

* Configure MINLP solver options
option minlp = dicopt;
$onecho > dicopt.opt
maxcycles 500
nlpsolver conopt3
mipsolver gurobi
$offecho
Reservoir.optfile = 1;

* Configure MIP solver options
option mip = Gurobi;
$onecho > gurobi.opt
MIPGap 5e-4
continue 1
$offecho
Reservoir.optfile = 1;

* Set general solution options
option sysout=on;
option optcr=0.0005;        // Optimality tolerance
OPTION ITERLIM = 1500000;   // Maximum iterations
option limrow=10, solprint=on;
OPTION reslim =1500000000;  // Maximum solution time - continue until optimal solution found

* ====================== VARIABLE INITIALIZATION ======================

* Initialize key variables with reasonable starting values
V.l(time_long,r_heyu) = V0(r_heyu);
V_end_t.l(time_long,r_heyu)= V0(r_heyu);
Q.l(time_long,r_heyu) = min(Qmax(r_heyu), max(Qmin(r_heyu), Inflow_total(time_long,r_heyu)));
Spill.l(time_long,r_heyu) = 0;
H_up.l(time_long,r_heyu) = H0(r_heyu);
H_down.l(time_long,r_heyu) = H_xia(r_heyu);
Inflow.l(time_long,r_heyu) = Inflow_total(time_long,r_heyu);
Q_small_A.l(time_long) = min(52, Q.l(time_long,'A'));
Q_large_A.l(time_long) = max(0, Q.l(time_long,'A') - 52);

* ====================== SOLVE THE MODEL ======================

* Solve the optimization problem to maximize total benefit
solve Reservoir using minlp maximizing Z;

* ====================== RESULTS OUTPUT ======================

* Save all relevant results to GDX file for post-processing
execute_unload 'results_both_lxb_and_rz-1.gdx',
     time_xun_or_day.l,
     Q.l, spill.l,Inflow.l,
     V.l, V_end_t.l, H_up.l, H_down.l,
     Compensatory_division.l,E_profit.l,Compensatory_division_fee.l
     E.l, N.l,
     abandon_or_not.l,Q_dy.l,
     Q_small_A.l, Q_large_A.l
     A_small_stage.l, A_large_stage.l
;

* Display final objective value and marginals
display Z.l,Z.m;